<?php
/**
 * BillDesk Payment Gateway API Endpoint
 * Handles payment order creation with JWE/JWS encryption
 * Returns JSON response
 * 
 * Deployed at: https://billapi.shakuniya.in/billdeskapi/api/create_order.php
 */

// Handle CORS
if (isset($_SERVER['HTTP_ORIGIN'])) {
    header("Access-Control-Allow-Origin: {$_SERVER['HTTP_ORIGIN']}");
    header('Access-Control-Allow-Credentials: true');
    header('Access-Control-Max-Age: 86400');    // cache for 1 day
}

if ($_SERVER['REQUEST_METHOD'] == 'OPTIONS') {
    if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_METHOD']))
        header("Access-Control-Allow-Methods: GET, POST, OPTIONS");         

    if (isset($_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']))
        header("Access-Control-Allow-Headers: {$_SERVER['HTTP_ACCESS_CONTROL_REQUEST_HEADERS']}");

    exit(0);
}

header('Content-Type: application/json');

// Suppress warnings to prevent JSON corruption
error_reporting(E_ALL & ~E_WARNING & ~E_NOTICE);
ini_set('display_errors', 0);

// Check for autoloader
if (!file_exists(__DIR__ . '../vendor/autoload.php')) {
    echo json_encode(['status' => 'error', 'message' => 'Vendor autoload not found. Please ensure the vendor folder is uploaded to the server.']);
    exit;
}

// Include Composer autoloader
require __DIR__ . '../vendor/autoload.php';

// Database Connection
$conn = null;

// Try to include global db.php
if (file_exists(__DIR__ . '../db.php')) {
    ob_start(); // Capture any accidental output
    include __DIR__ . '../db.php';
    ob_end_clean();
}

// Fallback connection if db.php didn't initialize $conn
if (!$conn || !($conn instanceof mysqli)) {
    $db_host = '127.0.0.1';
    $db_port = 3306;
    $db_user = 'vyaparbot_db';
    $db_password = 't(lRbhlAT%OvKJ7s';
    $db_name = 'vyaparbot_@server';
    
    $conn = new mysqli($db_host, $db_user, $db_password, $db_name, $db_port);
}

if ($conn->connect_error) {
    echo json_encode(['status' => 'error', 'message' => "Database connection failed: " . $conn->connect_error]);
    exit;
}
$conn->set_charset("utf8mb4");

// Retrieve form parameters from GET or POST or JSON
$rawInput = file_get_contents('php://input');
$input = json_decode($rawInput, true);

if (json_last_error() !== JSON_ERROR_NONE && !empty($rawInput)) {
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON input']);
    exit;
}

// ============================================================
// ENVIRONMENT CONFIGURATION
// ============================================================
$useUAT = isset($input['env']) && strtolower($input['env']) === 'uat';

if ($useUAT) {
    define('CREATE_ORDER_URL', 'https://uat1.billdesk.com/u2/payments/ve1_2/orders/create');
    define('PAYMENT_FORM_URL', 'https://uat1.billdesk.com/u2/web/v1_2/embeddedsdk');
} else {
    define('CREATE_ORDER_URL', 'https://api.billdesk.com/payments/ve1_2/orders/create');
    define('PAYMENT_FORM_URL', 'https://api.billdesk.com/web/v1_2/embeddedsdk');
}

// Use keys from payload if available, otherwise use defaults
$merchantId = $input['merchant_id'] ?? ($useUAT ? 'BDUAT2K526' : 'SHAKUNIYA');
$clientId   = $input['client_id']   ?? ($useUAT ? 'bduat2k526' : 'shakuniyasj');
$keyId      = $input['key_id']      ?? ($useUAT ? 'apLkPDnO9NYf' : 'l6SVGjSf5Gun');
$sharedKey  = $input['shared_key']  ?? ($useUAT ? 'xzYsZwfLeMDYzIweWT8i90QIWymCBXSn' : '17mAMIqwOBvS9YIQEzeP7VStbgvXUuWK');
$signingKey = $input['signing_key'] ?? ($useUAT ? 'RlKxmMPu6RB1BGaLKGxnnPXon8ynoSKh' : 'z1XIAONLs2YecHz1CYiVlysiJzVzT6h8');

$amount    = $input['tramt'] ?? ($_GET['tramt'] ?? ($_POST['tramt'] ?? 0));
$name      = $input['csname'] ?? ($_GET['csname'] ?? ($_POST['csname'] ?? ""));
$email     = $input['csemail'] ?? ($_GET['csemail'] ?? ($_POST['csemail'] ?? ""));
$number    = $input['csnumber'] ?? ($_GET['csnumber'] ?? ($_POST['csnumber'] ?? ""));
$address   = $input['csaddress'] ?? ($_GET['csaddress'] ?? ($_POST['csaddress'] ?? ""));
$productid = $input['prodid'] ?? ($_GET['prodid'] ?? ($_POST['prodid'] ?? 0));
$userId    = $input['user_id'] ?? 1; // Default to 1 if not provided
$planId    = $input['plan_id'] ?? ($productid ?: 1); // Map prodid to plan_id

// Determine Return URL dynamically
$protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http";
$host = $_SERVER['HTTP_HOST'];
// Assuming structure: /billdeskapi/api/create_order.php -> /billdeskapi/return.php
$basePath = dirname(dirname($_SERVER['SCRIPT_NAME'])); 
// Fix for Windows paths if running locally
$basePath = str_replace('\\', '/', $basePath);
$returnUrl = $protocol . "://" . $host . $basePath . '/return.php';

/**
 * Encode data to base64url format (RFC 4648)
 */
function base64url_encode(string $data): string {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

/**
 * Decode base64url to binary data
 */
function base64url_decode(string $data): string {
    $padded = $data . str_repeat('=', (4 - strlen($data) % 4) % 4);
    return base64_decode(strtr($padded, '-_', '+/'));
}

/**
 * Generate unique identifiers
 */
$orderId   = 'ORD' . time();
$traceid   = bin2hex(random_bytes(16));
$timestamp = time();

/**
 * Step 1: Insert order into database (New Schema)
 */
$stmt = $conn->prepare("INSERT INTO orders(user_id, subscription_plan_id, amount, pay_type, status, transaction_id, created_at, updated_at) VALUES (?, ?, ?, 'BILLDESK', 'pending', ?, NOW(), NOW())");

if ($stmt) {
    $stmt->bind_param('idss', $userId, $planId, $amount, $orderId);
    $stmt->execute();
    if ($stmt->error) {
        error_log("DB Insert Error: " . $stmt->error);
    }
    $stmt->close();
} else {
    error_log("DB Prepare Error: " . $conn->error);
}

/**
 * Step 2: Create JSON payload
 */
$payload = json_encode([
    'mercid'     => $merchantId,
    'orderid'    => $orderId,
    'amount'     => $amount . '.00',
    'order_date' => date('Y-m-d\TH:i:sP'),
    'currency'   => '356', // INR
    'ru'         => $returnUrl,
    'itemcode'   => 'DIRECT',
    'additional_info' => [
        'additional_info1' => $name,
        'additional_info2' => $email,
        'additional_info3' => $productid,
    ],
    'device'     => [
        'init_channel'  => 'internet',
        'ip'            => $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1',
        'user_agent'    => $_SERVER['HTTP_USER_AGENT'] ?? 'Mozilla/5.0',
    ],
]);

/**
 * Step 3: Create JWE (JSON Web Encryption)
 */
function createJWE(string $payload, string $key, string $keyId, string $clientId): string {
    $header = [
        'alg' => 'dir',
        'enc' => 'A256GCM',
        'kid' => $keyId,
        'clientid' => $clientId,
    ];
    
    $protectedHeader = base64url_encode(json_encode($header));
    $encryptedKey = '';
    $iv = openssl_random_pseudo_bytes(12);
    $aad = $protectedHeader;
    $tag = '';
    $ciphertext = openssl_encrypt(
        $payload,
        'aes-256-gcm',
        $key,
        OPENSSL_RAW_DATA,
        $iv,
        $tag,
        $aad
    );
    
    if ($ciphertext === false) {
        throw new Exception('JWE encryption failed: ' . openssl_error_string());
    }
    
    return implode('.', [
        $protectedHeader,
        base64url_encode($encryptedKey),
        base64url_encode($iv),
        base64url_encode($ciphertext),
        base64url_encode($tag)
    ]);
}

/**
 * Step 4: Create JWS (JSON Web Signature)
 */
function createJWS(string $payload, string $key, string $keyId, string $clientId): string {
    $header = [
        'alg' => 'HS256',
        'kid' => $keyId,
        'clientid' => $clientId,
    ];
    
    $headerEncoded = base64url_encode(json_encode($header));
    $payloadEncoded = base64url_encode($payload);
    $signingInput = $headerEncoded . '.' . $payloadEncoded;
    $signature = hash_hmac('sha256', $signingInput, $key, true);
    
    return $signingInput . '.' . base64url_encode($signature);
}

/**
 * Step 5: Verify JWS signature
 */
function verifyJWS(string $jws, string $key): string {
    $parts = explode('.', $jws);
    if (count($parts) !== 3) {
        throw new Exception('Invalid JWS format');
    }
    
    list($headerB64, $payloadB64, $signatureB64) = $parts;
    
    $signingInput = $headerB64 . '.' . $payloadB64;
    $signature = base64url_decode($signatureB64);
    $expectedSignature = hash_hmac('sha256', $signingInput, $key, true);
    
    if (!hash_equals($expectedSignature, $signature)) {
        throw new Exception('Invalid JWS signature');
    }
    
    return base64url_decode($payloadB64);
}

/**
 * Step 6: Decrypt JWE
 */
function decryptJWE(string $jwe, string $key): string {
    $parts = explode('.', $jwe);
    if (count($parts) !== 5) {
        throw new Exception('Invalid JWE format');
    }
    
    list($protectedHeader, $encryptedKeyB64, $ivB64, $ciphertextB64, $tagB64) = $parts;
    
    $iv = base64url_decode($ivB64);
    $ciphertext = base64url_decode($ciphertextB64);
    $tag = base64url_decode($tagB64);
    $aad = $protectedHeader;
    
    $plaintext = openssl_decrypt(
        $ciphertext,
        'aes-256-gcm',
        $key,
        OPENSSL_RAW_DATA,
        $iv,
        $tag,
        $aad
    );
    
    if ($plaintext === false) {
        throw new Exception('JWE decryption failed: ' . openssl_error_string());
    }
    
    return $plaintext;
}

try {
    // Create JWE from payload
    $jweToken = createJWE($payload, $sharedKey, $keyId, $clientId);

    // Sign the JWE with JWS
    $signedToken = createJWS($jweToken, $signingKey, $keyId, $clientId);

    /**
     * Step 7: Send request to BillDesk API
     */
    $ch = curl_init(CREATE_ORDER_URL);

    curl_setopt_array($ch, [
        CURLOPT_POST           => true,
        CURLOPT_POSTFIELDS     => $signedToken,
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_TIMEOUT        => 30,
        CURLOPT_HTTPHEADER     => [
            'Content-Type: application/jose',
            'Accept: application/jose',
            "BD-Traceid: $traceid",
            "BD-Timestamp: $timestamp"
        ],
    ]);

    $encryptedResponse = curl_exec($ch);

    if (curl_errno($ch)) {
        throw new Exception('cURL Error: ' . curl_error($ch));
    }

    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);

    // Validate response
    if (empty($encryptedResponse)) {
        throw new Exception("Empty response from BillDesk API. HTTP Code: " . $httpCode);
    }

    /**
     * Helper function to try to decrypt BillDesk response
     */
    function tryDecryptResponse($response, $signingKey, $sharedKey) {
        if (substr_count($response, '.') !== 2) return null;
        $parts = explode('.', $response);
        if (count($parts) !== 3) return null;
        
        try {
            $jwePayload = base64url_decode($parts[1]);
            if (substr_count($jwePayload, '.') !== 4) return null;
            
            $jweParts = explode('.', $jwePayload);
            if (count($jweParts) !== 5) return null;
            
            list($protectedHeader, $encryptedKeyB64, $ivB64, $ciphertextB64, $tagB64) = $jweParts;
            
            $iv = base64url_decode($ivB64);
            $ciphertext = base64url_decode($ciphertextB64);
            $tag = base64url_decode($tagB64);
            $aad = $protectedHeader;
            
            $plaintext = openssl_decrypt(
                $ciphertext,
                'aes-256-gcm',
                $sharedKey,
                OPENSSL_RAW_DATA,
                $iv,
                $tag,
                $aad
            );
            
            if ($plaintext === false) return null;
            return json_decode($plaintext, true);
        } catch (Exception $e) {
            return null;
        }
    }

    if ($httpCode !== 200 && $httpCode !== 201) {
        $decryptedError = tryDecryptResponse($encryptedResponse, $signingKey, $sharedKey);
        
        if ($decryptedError) {
            echo json_encode([
                'status' => 'error',
                'message' => $decryptedError['message'] ?? 'Unknown error',
                'error_code' => $decryptedError['error_code'] ?? $decryptedError['status'] ?? 'Unknown',
                'full_response' => $decryptedError
            ]);
            exit;
        }
        
        $errorData = json_decode($encryptedResponse, true);
        if ($errorData && isset($errorData['error_code'])) {
            echo json_encode([
                'status' => 'error',
                'message' => $errorData['message'] ?? 'Unknown error',
                'error_code' => $errorData['error_code']
            ]);
            exit;
        }
        
        echo json_encode([
            'status' => 'error',
            'message' => 'BillDesk API returned HTTP ' . $httpCode,
            'raw_response' => substr($encryptedResponse, 0, 200)
        ]);
        exit;
    }

    /**
     * Step 8: Decrypt and verify response
     */
    $paymentLink = '';
    $rdata = '';
    $bdorderid = '';

    // First check if response is HTML
    if (strpos($encryptedResponse, '<html') !== false || strpos($encryptedResponse, '<!DOCTYPE') !== false) {
        throw new Exception('BillDesk returned HTML error page - WAF rejection or server error.');
    }
    
    // Check if response looks like a JWS token
    if (substr_count($encryptedResponse, '.') < 2) {
        $responseData = json_decode($encryptedResponse, true);
        if ($responseData && isset($responseData['payment_link'])) {
            $paymentLink = $responseData['payment_link'];
        } else {
            throw new Exception('Invalid response format: expected JWS token or JSON with payment_link');
        }
    } else {
        $jwePayload = verifyJWS($encryptedResponse, $signingKey);
        $decryptedPayload = decryptJWE($jwePayload, $sharedKey);
        $data = json_decode($decryptedPayload, true);
        
        if (!is_array($data)) {
            throw new Exception('Invalid JSON in decrypted payload');
        }
        
        if (isset($data['links']) && is_array($data['links']) && count($data['links']) > 1) {
            $rdata = $data['links'][1]['parameters']['rdata'] ?? '';
            $bdorderid = $data['links'][1]['parameters']['bdorderid'] ?? '';
            
            // Update database with BillDesk Order ID (New Schema)
            $updateStmt = $conn->prepare("UPDATE orders SET bd_order_id = ?, gateway_payload = ? WHERE transaction_id = ?");
            if ($updateStmt) {
                $payloadJson = json_encode($data);
                $updateStmt->bind_param('sss', $bdorderid, $payloadJson, $orderId);
                $updateStmt->execute();
                $updateStmt->close();
            }
            
            $paymentLink = $data['links'][1]['uri'] ?? $data['links'][1]['url'] ?? '';
        } else {
            throw new Exception('Unexpected response structure from BillDesk');
        }
    }
    
    if (empty($paymentLink) && !empty($rdata)) {
        echo json_encode([
            'status' => 'success',
            'type' => 'form_submit',
            'url' => PAYMENT_FORM_URL,
            'bdorderid' => $bdorderid,
            'merchantid' => $merchantId,
            'rdata' => $rdata
        ]);
        exit;
    }
    
    if (empty($paymentLink)) {
        throw new Exception('No payment link extracted from response');
    }

    echo json_encode([
        'status' => 'success',
        'type' => 'redirect',
        'url' => $paymentLink
    ]);

} catch (Exception $e) {
    echo json_encode([
        'status' => 'error',
        'message' => $e->getMessage()
    ]);
}
