<?php
/**
 * Base64URL‐encode.
 */
function base64url_encode(string $data): string {
    return rtrim(strtr(base64_encode($data), '+/', '-_'), '=');
}

/**
 * Create a JWE Compact Serialization with direct AES‑256‑GCM.
 *
 * @param string $plaintext      The JSON (or plain) payload to encrypt.
 * @param string $encryptionKey  32‑byte raw key.
 * @param string $keyId          BillDesk encryption key ID.
 * @param string $clientId       BillDesk client ID.
 * @return string                JWE Compact (5 parts, dot‑separated).
 */
function jwe_encrypt_dir_a256gcm(
    string $plaintext,
    string $encryptionKey,
    string $keyId,
    string $clientId
): string {
    // 1) Build protected header
    $hdr = [
        'alg'      => 'RS256',
        'enc'      => 'A256GCM',
        'kid'      => $keyId,
        'clientid' => $clientId
    ];
    $protected = base64url_encode(json_encode($hdr));

    // 2) Generate a 96‑bit IV
    $iv = random_bytes(12);

    // 3) Encrypt with AES‑256‑GCM, passing the protected header as AAD
    $tag = '';
    $ciphertext = openssl_encrypt(
        $plaintext,
        'aes-256-gcm',
        $encryptionKey,
        OPENSSL_RAW_DATA,
        $iv,
        $tag,
        $protected,
        16   // 128‑bit tag
    );

    // 4) Assemble the five‐part compact JWE: [protected].[encryptedKey].[iv].[ciphertext].[tag]
    //    For "dir" encryptedKey is empty
    return implode('.', [
        $protected,
        '',
        base64url_encode($iv),
        base64url_encode($ciphertext),
        base64url_encode($tag)
    ]);
}

/**
 * Create a JWS Compact Serialization (HMAC).
 *
 * @param string $payload     The JSON (or plain) payload to sign.
 * @param string $signingKey  Raw HMAC key.
 * @param string $keyId       BillDesk signing key ID.
 * @param string $clientId    BillDesk client ID.
 * @param string $alg         One of "HS256","HS384","HS512".
 * @return string             JWS Compact (3 parts).
 */
function jws_sign_hs(
    string $payload,
    string $signingKey,
    string $keyId,
    string $clientId,
    string $alg = 'HS256'
): string {
    // map alg → hash algorithm
    $hashes = ['HS256'=>'sha256','HS384'=>'sha384','HS512'=>'sha512'];
    if (!isset($hashes[$alg])) {
        throw new \InvalidArgumentException("Unsupported alg $alg");
    }

    // 1) Header
    $hdr = [
        'alg'      => $alg,
        'kid'      => $keyId,
        'clientid' => $clientId
    ];
    $protected = base64url_encode(json_encode($hdr));

    // 2) Message to sign
    $message = $protected . '.' . base64url_encode($payload);

    // 3) Compute HMAC
    $sig = hash_hmac($hashes[$alg], $message, $signingKey, true);

    // 4) Compact serialization: [protected].[payload].[signature]
    return $message . '.' . base64url_encode($sig);
}

// ─── USAGE ────────────────────────────────────────────────────────────────────



const CREATE_ORDER_URL = 'https://uat1.billdesk.com/u2/payments/ve1_2/orders/create';


// === 1) BUILD YOUR ORDER PAYLOAD ===
$payload = json_encode([
    'mercid'     => 'BDUAT2K526',
    'orderid'    => 'ORD' . time(),
    'amount'     => '100.00',
    'order_date' => date('Y-m-d\TH:i:sP'),
    'currency'   => '356',
    'ru'         => 'https://shakuniya.ub.com/return',
    'itemcode'   => 'DIRECT',
    'device'     => [
        'init_channel'             => 'internet',
        'ip'                       => $_SERVER['REMOTE_ADDR'] ?? '',
        'user_agent'               => $_SERVER['HTTP_USER_AGENT'] ?? '',
        'accept_header'            => $_SERVER['HTTP_ACCEPT'] ?? '',
        'browser_tz'               => (new DateTime())->getOffset() / 60,
        'browser_color_depth'      => '24',
        'browser_java_enabled'     => 'false',
        'browser_screen_height'    => '',
        'browser_screen_width'     => '',
        'browser_language'         => $_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? '',
        'browser_javascript_enabled'=> 'true',
    ],
]);

$responseString = '{"order":"12345","amount":100}';   // whatever you need to encrypt
$encryptionKey   = hex2bin('xzYsZwfLeMDYzIweWT8i90QIWymCBXSn'); 
$keyIdEnc        = 'apLkPDnO9NYf';
$clientId        = 'bduat2k526';

$jwe = jwe_encrypt_dir_a256gcm(
    $payload,
    $encryptionKey,
    $keyIdEnc,
    $clientId
);
// $jwe now holds your JWE compact string
echo "jwe: $jwe";
echo "</br>";
$signingKey     = hex2bin('RlKxmMPu6RB1BGaLKGxnnPXon8ynoSKh');
$keyIdSign      = 'apLkPDnO9NYf';

$encryptedRequest = jws_sign_hs(
    $jwe,
    $signingKey,
    $keyIdSign,
    $clientId,
    'HS256'
);
echo "jsw: $encryptedRequest";

$ch = curl_init(CREATE_ORDER_URL);
$traceid   = bin2hex(random_bytes(16));
$timestamp = time();
curl_setopt_array($ch, [
    CURLOPT_POST           => true,
    CURLOPT_POSTFIELDS     => $encryptedRequest,
    CURLOPT_RETURNTRANSFER => true,
    CURLOPT_HTTPHEADER     => [
        'Content-Type: application/jose',
        'Accept: application/jose',
        "BD-Traceid: $traceid",
        "BD-Timestamp: $timestamp"
    ],
]);
$encryptedResponse = curl_exec($ch);
if (curl_errno($ch)) {
    die('cURL Error: ' . curl_error($ch));
}
curl_close($ch);
echo "response: ".$encryptedResponse;

// $jws now holds your JWS compact string
